using System;
using System.Xml;

using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages
{
	/// <summary>
	/// Partial success logon response message requiring client to 
	/// select a division. 
	/// </summary>
	public sealed class SecurityLogonDivisionRequiredResponseMessage : SecurityBasePartialSuccessResponseMessage
	{
		/// <summary>
		/// VistALink message response type string. 
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string ResponseType = "AV.Logon";
		
		private DivisionInfoCollection _divisions;
		private string _postSignInText;

		/// <summary>
		/// Constructor that needs divisions list and a post sign-in text to create a message.
		/// </summary>
		/// <param name="divisions">List of divisions available for user to select. Required.</param>
		/// <param name="postSignInText">Server post sign-in text message. Optional - may be null.</param>
		public SecurityLogonDivisionRequiredResponseMessage( DivisionInfoCollection divisions, string postSignInText )
			: this( divisions, postSignInText, null ) {}

		/// <summary>
		/// Constructor specifying divisions list, post sign-in text and informative message.
		/// </summary>
		/// <param name="divisions">List of divisions available for user to select. Required.</param>
		/// <param name="postSignInText">Server post sign-in text message. Optional - may be null.</param>
		/// <param name="message">Informative message. Optional - may be null.</param>
		public SecurityLogonDivisionRequiredResponseMessage( DivisionInfoCollection divisions, string postSignInText, string message )
			: base( ResponseType, message )
		{
			if( divisions == null ) 
				throw( new ArgumentNullException( "divisions" ) );

			if( divisions.Count < 2 ) 
				throw( new ArgumentOutOfRangeException( SR.Exceptions.DivisionListAtLeastTwoEntriesRequired() ) );

			_divisions = divisions;
			_postSignInText = postSignInText;
		}

		/// <summary>
		/// Standard VistALink message deserialization constructor. 
		/// Allows recreating message object from XML document.
		/// </summary>
		/// <param name="sourceDoc">Source XML document.</param>
		public SecurityLogonDivisionRequiredResponseMessage( XmlDocument sourceDoc )
			: base( ResponseType, sourceDoc )
		{
			_postSignInText = 
				SecurityLogonNonFailureResponseMessageImplementationMixin.ParsePostSignInTextXml( XmlParseGetResponseElement( sourceDoc ), false );

			_divisions = DivisionInfoCollection.Parse( XmlParseGetPartialSuccessElement( sourceDoc ) );

			if( _divisions.Count < 2 )
				throw( new XmlParseException( SR.Exceptions.DivisionListAtLeastTwoEntriesRequired() ) );
		}

		/// <summary>
		/// Overriden member addind divisions and post sign-in text information to 
		/// the partial success element created by the base class. 
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		protected override void WriteXmlPartialSuccessNodeContent( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			base.WriteXmlPartialSuccessNodeContent( writer );

			_divisions.WriteDivisionsListToXml( writer );
		}

		/// <summary>
		/// Overriden method adding message-specific content to the response element
		/// created by the base class. 
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		protected override void WriteXmlResponseNodeContent( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			SecurityLogonNonFailureResponseMessageImplementationMixin.WritePostSignInTextToXml( writer, _postSignInText, false );

			base.WriteXmlResponseNodeContent( writer );
		}


		/// <summary>
		/// Utility XML parsing method utilized to find out whether the supplied VistALink partial success response 
		/// message in XML document partial success type matches this specific message type. It's used in message-to-XML mapping, 
		/// when parsing server XML response and creating an appropriate instance of VistALink message. 
		/// Beware, that source XML document must have 'partialSuccess' element in proper place. Otherwise, XML parsing 
		/// exception will be thrown.  
		/// </summary>
		/// <param name="sourceDoc">Source XML document to analyze.</param>
		/// <returns>True if partial success element in the source XML document contains divisions data.</returns>
		internal static bool XmlParseDoesPartialSuccessXmlHaveDivisionsData( XmlDocument sourceDoc )
		{
			if( sourceDoc == null )
				throw( new ArgumentNullException( "sourceDoc" ) );

			XmlElement _partialSuccessElement = SecurityBasePartialSuccessResponseMessage.XmlParseGetPartialSuccessElement( sourceDoc );

			return DivisionInfoCollection.XmlParseIsValidParentElement( _partialSuccessElement );
		}
		
		/// <summary>
		/// List of divisions available for user to select from.
		/// </summary>
		public DivisionInfoCollection Divisions
		{
			get
			{
				return _divisions;
			}
		}

		/// <summary>
		/// Server post sign-in text that should be shown to user. 
		/// </summary>
		public string PostSignInText
		{
			get
			{
				return _postSignInText;
			}
		}
	}
}
